import { Injectable } from '@nestjs/common';
import { GraphService } from './graph.service';
import {
  Graph,
  GraphId,
  GraphNode,
  GraphNodeId,
  PathType,
  NearestReturnType,
} from './navigation.entity';
import { PathfindService } from './pathfind.service';

@Injectable()
export class NavigationService {
  constructor(
    private readonly graphService: GraphService,
    private readonly pathfindService: PathfindService,
  ) {}

  /**
   * Get a graph by its ID.
   * @param id The ID of the graph.
   * @returns The graph object or null if not found.
   */
  async getGraph(id: GraphId): Promise<Graph | null> {
    return await this.graphService.getGraph(id);
  }

  /**
   * Get a node by its ID.
   * @param id The ID of the node.
   * @returns The graph node object or null if not found.
   */
  async getNode(id: GraphNodeId): Promise<GraphNode | null> {
    return await this.graphService.getNode(id);
  }

  /**
   * Get a path between two nodes using A* algorithm.
   * @param from The starting node ID.
   * @param to The destination node ID.
   * @param type The type of path to return (shallow or deep).
   * @returns An array of GraphNode representing the path, or an array of string IDs if shallow.
   */
  async getPath(
    from: GraphNodeId,
    to: GraphNodeId,
    type: PathType,
  ): Promise<GraphNode[] | { nodes: GraphNodeId[] }> {
    // get nodes from the graph service
    const fromNode = await this.graphService.getNode(from);
    const toNode = await this.graphService.getNode(to);
    if (!fromNode || !toNode) {
      throw new Error('Invalid nodes for pathfinding');
    }

    // ensure both nodes belong to the same graph
    if (fromNode.graphId !== toNode.graphId) {
      throw new Error('Nodes must belong to the same graph for pathfinding');
    }

    // get the graph from the graph service
    const graph = await this.graphService.getGraph(fromNode.graphId);
    if (!graph) {
      throw new Error('Graph not found for pathfinding');
    }

    // perform A* pathfinding using the pathfind service
    const path = this.pathfindService.AStar(graph, fromNode, toNode);
    if (!path) {
      throw new Error('Path not found');
    }

    if (type === PathType.SHALLOW) {
      // only return the id of each node
      return { nodes: path.map((node) => node.id) };
    }
    return path;
  }

  /**
   * Get the nearest node to the given coordinates.
   * @param x The x-coordinate.
   * @param y The y-coordinate.
   * @param z The z-coordinate.
   * @param yThreshold The y-coordinate threshold for filtering nodes.
   * @param returnType The type of return value (id or full node).
   * @returns The ID of the nearest node or the full graph node object.
   */
  async getNearestNode(
    x: number,
    y: number,
    z: number,
    yThreshold?: number,
    returnType: NearestReturnType = NearestReturnType.SHALLOW,
  ): Promise<string | GraphNode | null> {
    return await this.graphService.getNearestNode(
      x,
      y,
      z,
      yThreshold,
      returnType,
    );
  }
}
